#Rem
	Header: Collision Mask Module V1.0
	By David Powell
	[a http://www.loadcode.co.uk]http://www.loadcode.co.uk[/a]
	Released into the public domain
#End

#BINARY_FILES += "*.cmsk"

Strict

Import mojo 'Only required for DebugDraw method

Import brl.databuffer
Import brl.datastream

Public
	#Rem
		Summary: A simple mask, which cannot be scaled or rotated.
		Scale and rotate methods on MasterMasks will return this simple Mask.
	#End
	Class Mask
		Public
			'Summary: Default mask flags (no flags set)
			Const DefaultFlags:Int = 0
			'Summary: Set the mask handle to the middle of the mask
			Const MidHandle:Int = 1

		Private
			Field width:Int
			Field height:Int
			Field flags:Int
			Field handleX:Float
			Field handleY:Float
			Field threshold:Int = 128
			Field scaleX:Float = 1.0
			Field scaleY:Float = 1.0
			Field rotation:Float = 0.0
			Field path:String
			Field data:Int[]
			Field onComplete:IOnLoadMaskComplete
			
		Public
			'Summary: Get the width of the mask.
			Method Width:Int() Property
				Return width
			End
			
		Public
			'Summary: Get theight of the mask.
			Method Height:Int() Property
				Return height
			End
			
		Public
			'Summary: Get the X position of the handle.
			Method HandleX:Float() Property
				Return handleX
			End
			
		Private
			'HandleX is read-only in the base class.
			Method HandleX:Void(value:Float) Property
				Error("HandleX is read-only in the Mask base class")
			End
			
		Public
			'Summary: Get the Y position of the handle.
			Method HandleY:Float() Property
				Return handleY
			End
			
		Private
			'HandleY is read-only in the base class.
			Method HandleY:Void(value:Float) Property
				Error("HandleY is read-only in the Mask base class")
			End
			
		Public
			'Summary: Get the collision threshold.
			Method Threshold:Int() Property
				Return threshold
			End
			
		Public
			'Summary: Set the collision threshold.
			Method Threshold:Void(value:Int) Property
				threshold = value
			End
			
		Public
			'Summary: Get the X scale.
			Method ScaleX:Float() Property
				Return scaleX
			End
			
		Public
			'Summary: Get the Y scale.
			Method ScaleY:Float() Property
				Return scaleY
			End
			
		Public
			'Summary: Get the rotation angle.
			Method Rotation:Float() Property
				Return rotation
			End
		
		Public
			#Rem
				Summary: Draw the mask on the screen for debugging purposes
				[i]x[/i] : The X position to draw the mask, offset by the masks handle
				[i]y[/i] : The Y position to draw the mask, offset by the masks handle
				[i]alpha[/i] : The alpha level to draw the mask
				[i]red[/i] : The colour to draw the mask, red component
				[i]green[/i] : The colour to draw the mask, green component
				[i]blue[/i] : The colour to draw the mask, blue component
				
				This method should only be used for debugging as it is very slow.
			#End
			Method DebugDraw:Void(x:Int, y:Int, alpha:Float = 1.0, red:Float = 255, green:Float = 255, blue:Float = 255)
				Local oldAlpha:Float = GetAlpha()
				Local oldColour:Float[] = GetColor()
				Local redWeight:Float = red / 255
				Local greenWeight:Float = green / 255
				Local blueWeight:Float = blue / 255
				Local verticalOffset:Int = 0
				
				Local drawX:Int = x - handleX
				Local drawY:Int = y - handleY
				
				SetAlpha(alpha)
				For Local maskY:Int = 0 Until height
					For Local maskX:Int = 0 Until width
						Local maskValue:Int = data[verticalOffset + maskX]
						SetColor(maskValue * redWeight, maskValue * greenWeight, maskValue * blueWeight)
						DrawPoint(drawX + maskX, drawY + maskY)
					Next
					verticalOffset += width
				Next
				SetColor(oldColour[0], oldColour[1], oldColour[2])
				SetAlpha(oldAlpha)
			End
			
		Public
			#Rem
				Summary: Check to see if this mask collides with another mask when at the specified locations
				[i]maskX[/i] : The X position of this mask, offset by the masks handle
				[i]maskY[/i] : The Y position of this mask, offset by the masks handle
				[i]mask2[/i] : A second mask to check against this one
				[i]mask2X[/i] : The X position of the second mask, offset by the masks handle
				[i]mask2Y[/i] : The Y position of the second mask, offset by the masks handle
				[i]Returns[/i] : True if the masks collide, False otherwise 
			#End
			Method CollideMask:Bool(maskX:Int, maskY:Int, mask2:Mask, mask2X:Int, mask2Y:Int)
				'Calculate intersecting rect
				Local maskPos1X:Int = maskX - Self.handleX 'X position of top-left of mask 1 (Self)
				Local maskPos1Y:Int = maskY - Self.handleY 'Y position of top-left of mask 1 (Self)
				Local maskPos2X:Int = mask2X - mask2.handleX 'X position of top-left of mask 2
				Local maskPos2Y:Int = mask2Y - mask2.handleY 'Y position of top-left of mask 2
				
				Local x1:Int = Max(maskPos1X, maskPos2X) 'X position of top-left of rectangle intersecting the two masks
				Local y1:Int = Max(maskPos1Y, maskPos2Y) 'Y position of top-left of rectangle intersecting the two masks
				Local x2:Int = Min(maskPos1X + Self.width, maskPos2X + mask2.width) 'X position of bottom-right of rectangle intersecting the two masks
				Local y2:Int = Min(maskPos1Y + Self.height, maskPos2Y + mask2.height) 'Y position of bottom-right of rectangle intersecting the two masks
				If x2 < x1 Or y2 < y1
					'Rectangles do not intersect
		  			Return False
				EndIf
				
				Local intersectWidth:Int = x2 - x1 'The width of the intersecting rectangle
				Local intersectHeight:Int = y2 - y1  'The height of the intersecting rectangle
				
				Local mask1Offset:Int = ( (y1 - maskPos1Y) * Self.width) + (x1 - maskPos1X) 'Offset into mask 1 (Self) array of top-left of intersection rectangle
				Local mask2Offset:Int = ( (y1 - maskPos2Y) * mask2.width) + (x1 - maskPos2X) 'Offset into mask 2 array of top-left of intersection rectangle
				
				'Check each point in the intersecting rectangle to see if there is a collision
				For Local checkY:Int = 0 Until intersectHeight
					For Local checkX:Int = 0 Until intersectWidth
						If Self.data[mask1Offset + checkX] >= Self.threshold And mask2.data[mask2Offset + checkX] >= mask2.threshold  'Check if the masks collide
							Return True
						EndIf
					Next
					mask1Offset += Self.width 'Move to the next line of mask 1 (Self)
					mask2Offset += mask2.width 'Move to the next line of mask 2
				Next
				
				Return False
			End
			
		Public
			#Rem
				Summary: Check to see if this mask collides with a rectangle
				[i]maskX[/i] : The X position of the mask, offset by the masks handle
				[i]maskY[/i] : The Y position of the mask, offset by the masks handle
				[i]rectX[/i] : The X position of to top-left of the rectangle
				[i]rectY[/i] : The Y position of to top-left of the rectangle
				[i]rectWidth[/i] : The width of the rectangle
				[i]rectHeight[/i] : The height of the rectangle
				[i]Returns[/i] : True if the mask collides with the rectangle, False otherwise
			#End
			Method CollideRect:Bool(maskX:Int, maskY:Int, rectX:Int, rectY:Int, rectWidth:Int, rectHeight:Int)
				'Calculate intersecting rect
				Local maskPosX:Int = maskX - Self.handleX 'X position of top-left of mask
				Local maskPosY:Int = maskY - Self.handleY 'Y position of top-left of mask
				
				Local x1:Int = Max(maskPosX, rectX) 'X position of top-left of rectangle intersecting mask and user specified rectangle
				Local y1:Int = Max(maskPosY, rectY) 'Y position of top-left of rectangle intersecting mask and user specified rectangle
				Local x2:Int = Min(maskPosX + Self.width, rectX + rectWidth) 'X position of bottom-right of rectangle intersecting mask and user specified rectangle
				Local y2:Int = Min(maskPosY + Self.height, rectY + rectHeight) 'Y position of bottom-right of rectangle intersecting mask and user specified rectangle
				If x2 < x1 Or y2 < y1
					'Rectangles do not intersect
		  			Return False
				EndIf
			
				Local intersectWidth:Int = x2 - x1 'The width of the intersecting rectangle
				Local intersectHeight:Int = y2 - y1 'The height of the intersecting rectangle
				
				Local maskOffset:Int = ( (y1 - maskPosY) * Self.width) + (x1 - maskPosX) 'Offset into mask array of top-left of intersection rectangle
				
				'Check each point in the intersecting rectangle to see if there is a collision
				For Local checkY:Int = 0 Until intersectHeight
					For Local checkX:Int = 0 Until intersectWidth
						If Self.data[maskOffset + checkX] >= Self.threshold 'Check if the mask should collide
							Return True
						EndIf
					Next
					maskOffset += Self.width 'Move to the next line of the mask
				Next
				
				Return False
			End
			
		Public
			#Rem
				Summary: Check to see if this mask collides with a point
				[i]maskX[/i] : The X position of the mask, offset by the masks handle
				[i]maskY[/i] : The Y position of the mask, offset by the masks handle
				[i]pointX[/i] : The X position of the point
				[i]pointY[/i] : The Y position of the point
				[i]Returns[/i] : True if the mask collides with the point, False otherwise
			#End
			Method CollidePoint:Bool(maskX:Int, maskY:Int, pointX:Int, pointY:Int)
				Local maskPosX:Int = maskX - Self.handleX 'X position of top-left of mask
				Local maskPosY:Int = maskY - Self.handleY 'Y position of top-left of mask
				
				'Check to see if the point is within the mask
				If (maskPosX <= pointX And pointX < maskPosX + Self.width And maskPosY <= pointY And pointY < maskPosY + Self.height) = False
					Return False
				EndIf
			
				Local maskOffset:Int = ( (pointY - maskPosY) * Self.width) + (pointX - maskPosX) 'Offset into mask array
				
				If Self.data[maskOffset] >= Self.threshold 'Check if the mask should collide
					Return True
				EndIf
				
				Return False
			End
			
		Public
			#Rem
				Summary: Check to see if this mask collides with a circle
				[i]maskX[/i] : The X position of the mask, offset by the masks handle
				[i]maskY[/i] : The Y position of the mask, offset by the masks handle
				[i]centreX[/i] : The X position of the centre of the circle
				[i]centreY[/i] : The Y position of the centre of the circle
				[i]radius[/i] : The radius of the circle
				[i]Returns[/i] : True if the mask collides with the circle, False otherwise
				
				Please note that the circle calculation used in this collision check uses a different algorithm to
				that used by the Mojo DrawCircle() function. Therefore, this collision will not be pixel perfect against it.
			#End
			Method CollideCircle:Bool(maskX:Int, maskY:Int, centreX:Int, centreY:Int, radius:Int)
				'Calculate intersecting rect
				Local maskPosX:Int = maskX - Self.handleX 'X position of top-left of mask
				Local maskPosY:Int = maskY - Self.handleY 'Y position of top-left of mask
				
				Local x1:Int = Max(maskPosX, centreX - radius) 'X position of top-left of rectangle intersecting virtual rectangle surrounding circle
				Local y1:Int = Max(maskPosY, centreY - radius) 'Y position of top-left of rectangle intersecting virtualrectangle surrounding circle
				Local x2:Int = Min(maskPosX + Self.width, centreX + radius) 'X position of bottom-right of rectangle intersecting virtual rectangle surrounding circle
				Local y2:Int = Min(maskPosY + Self.height, centreY + radius) 'Y position of bottom-right of rectangle intersecting virtual rectangle surrounding circle
				If x2 < x1 Or y2 < y1
					'Rectangles do not intersect
		  			Return False
				EndIf
				
				Local intersectWidth:Int = x2 - x1 'The width of the intersecting rectangle
				Local intersectHeight:Int = y2 - y1 'The height of the intersecting rectangle
				
				Local maskOffset:Int = ( (y1 - maskPosY) * Self.width) + (x1 - maskPosX) 'Offset into mask array of top-left of intersection rectangle
				Local xOffset:Int = x1 - centreX 'X offset into circle of top-left of rectangle intersection
				Local yOffset:Int = y1 - centreY 'Y offset into circle of top-left of rectangle intersection
				Local radiusSquared:Int = radius * radius 'The radius squared for use in Pythagorean hypotenuse calculation
				
				'Check each point in the intersecting rectangle to see if there is a collision
				For Local checkY:Int = 0 Until intersectHeight
					For Local checkX:Int = 0 Until intersectWidth
						If (xOffset + checkX) * (xOffset + checkX) + (yOffset + checkY) * (yOffset + checkY) <= radiusSquared 'Check if the point is within the circle (Pythagorean hypotenuse)
							If Self.data[maskOffset + checkX] >= Self.threshold 'Check if the mask should collide
								Return True
							EndIf
						EndIf
					Next
					maskOffset += Self.width 'Move to the next line of the mask
				Next
				
				Return False
			End
				
	End

Public
	#Rem
		Summary: Load a CMSK file and return a MasterMask representing it.
		[i]path[/i] : The full path of the CMSK file
		[i]flags[/i] : The mask flags to use
		[i]Returns[/i] : A MasterMask object representing the loaded CMSK file
		
		Please note that New is called in this function, therefore it should not be used within a main game loop.
	#End
	Function LoadMask:MasterMask(path:String, flags:Int = Mask.DefaultFlags)
		Local buffer:DataBuffer = DataBuffer.Load(path)
		Local mask:MasterMask = New MasterMask()
		mask.flags = flags
		mask.CreateMasterMask(buffer, path)
		Return mask
	End

Public
	#Rem
		Summary: Begin loading a CMSK file asynchronously
		[i]path[/i] : The full path of the CMSK file
		[i]flags[/i] : The mask flags to use
		[i]onComplete[/i] : The object to invoke the OnLoadMaskComplete method on when the mask has loaded
		
		As with all asynchronous loading, UpdateAsyncEvents() should be called repeatedly until
		the mask has been loaded.
		
		Please note that New is called in this function, therefore it should not be used within a main game loop.
	#End
	Function LoadMaskAsync:Void(path:String, flags:Int = Mask.DefaultFlags, onComplete:IOnLoadMaskComplete)
		Local mask:MasterMask = New MasterMask()
		mask.onComplete = onComplete
		mask.flags = flags
		DataBuffer.LoadAsync(path, mask)
	End

Public
	'Summary: A master mask, which allows scaling and rotation.
	Class MasterMask Extends Mask Implements IOnLoadDataComplete
		Private
			Const VersionOffset:Int = 10
			Const DataOffsetOffset:Int = 14
			Const DataLengthOffset:Int = 18
			Const WidthOffset:Int = 22
			Const HeightOffset:Int = 26

		Public
			'Summary: Set the X position of the handle.
			Method HandleX:Void(value:Float) Property
				Self.handleX = value
			End
			
		Public
			'Summary: Set the Y position of the handle.
			Method HandleY:Void(value:Float) Property
				Self.handleY = value
			End

		Public
			#Rem
				Summary: Set the X and Y position of the handle.
				[i]tx[/i] : The X position of the handle
				[i]ty[/i] : The Y position of the handle
			#End
			Method SetHandle:Void(tx:Float, ty:Float)
				Self.handleX = tx
				Self.handleY = ty
			End
		
		Private
			'Internal: When the data buffer has been loaded asynchronously create the mask.
			Method OnLoadDataComplete:Void(buffer:DataBuffer, path:String)			
				CreateMasterMask(buffer, path)
				Self.onComplete.OnLoadMaskComplete(Self, path)
				Self.onComplete = Null
			End
			
		Private
			'Internal: Create a master mask from the supplied data buffer.
			Method CreateMasterMask:Void(buffer:DataBuffer, path:String)
				If buffer = Null Then Error("The mask file '" + path + "' could not be loaded.")
				If CheckSignature(buffer) = False Then Error("The file '" + path + "' is not a mask, or the file is corrupt.")
				
				'If little endian
				#If TARGET <> "flash"
					SwitchIntEndianness(buffer, VersionOffset)
				#EndIF
				'EndIf
				Local version:Int = buffer.PeekInt(VersionOffset)
				If version <> 1 Then Error("The mask file '" + path + "' cannot be used as only version 1 CMSK files are supported.")
				
				FixEndian(buffer)
				
				Local dataOffset:Int = buffer.PeekInt(DataOffsetOffset)
				Local dataLength:Int = buffer.PeekInt(DataLengthOffset)
				Self.width = buffer.PeekInt(WidthOffset)
				Self.height = buffer.PeekInt(HeightOffset)

				Self.data = RleDecode(buffer, dataOffset, dataLength, Self.width * Self.height)
				
				If Self.flags & Mask.MidHandle <> 0
					SetHandle(Self.width / 2.0, Self.height / 2.0)
				EndIf
				
				Self.path = path
			End
				
		Private
			'Internal: Check the CMSK signature to check the file type is correct and not corrupted.
			Method CheckSignature:Bool(buffer:DataBuffer)
				'Due to signed/unsigned issue the first byte should be 139, but is read as -117
				Local signature:Int[] =[-117, 67, 77, 83, 75, 13, 10, 26, 10, 0]
				
				Local readSignature:Int[] = buffer.PeekBytes(0, 10)

				For Local offset:Int = 0 To 9
					If signature[offset] <> readSignature[offset]
						Return False
					EndIf
				Next
			
				Return True
			End
			
		Private
			'Internal: Fix the endianness of the integers in the CMSK file.
			Method FixEndian:Void(buffer:DataBuffer)
				'If little endian
				#If TARGET <> "flash"
					SwitchIntEndianness(buffer, DataOffsetOffset)
					SwitchIntEndianness(buffer, DataLengthOffset)
					SwitchIntEndianness(buffer, WidthOffset)
					SwitchIntEndianness(buffer, HeightOffset)
				#EndIF
				'EndIf
			End
			
		Private
			'Internal: Switch the endianness of an integer in a databuffer.
			Method SwitchIntEndianness:Void(buffer:DataBuffer, address:Int)
				Local temp:Int
				Local intBytes:Int[] = buffer.PeekBytes(address, 4)
				temp = intBytes[3]
				intBytes[3] = intBytes[0]
				intBytes[0] = temp
				temp = intBytes[2]
				intBytes[2] = intBytes[1]
				intBytes[1] = temp
				buffer.PokeBytes(address, intBytes, 0, 4)
			End
			
		Private
			'Internal: Decode the RLE data in the CMSK file.
			Method RleDecode:Int[] (buffer:DataBuffer, address:Int, length:Int, decodedLength:Int)
				Local stream:DataStream = New DataStream(buffer, address, length)
				Local data:Int[decodedLength]
				Local offset:Int = 0
				
				Local lastPixel:Int = stream.ReadByte()
				If lastPixel < 0
					lastPixel += 256 'Fix for signed read of unsigned byte
				EndIf
				
				While stream.Eof() = 0
					Local pixel:Int = stream.ReadByte()
					If pixel < 0
						pixel += 256 'Fix for signed read of unsigned byte
					EndIf
					
					If pixel <> lastPixel
						data[offset] = lastPixel
						offset += 1
					
						lastPixel = pixel
					Else
						Local count:Int = stream.ReadByte()
						If count < 0
							count += 256 'Fix for signed read of unsigned byte
						EndIf
					
						For Local num:Int = 0 Until count
							data[offset] = pixel
							offset += 1
						Next
						
						If stream.Eof() = 0
							lastPixel = stream.ReadByte()
							If lastPixel < 0
								lastPixel += 256 'Fix for signed read of unsigned byte
							EndIf
						EndIf
					EndIf
				Wend
				
				If offset < decodedLength
					data[offset] = lastPixel
					offset += 1
				EndIf
				
				If offset <> decodedLength
					Error("The mask file is corrupt.")
				EndIf
				
				Return data
			End
			
		Public
			#Rem
				Summary: Scale the mask by the specified X and Y amounts
				[i]scaleX[/i] : The amount to scale by horizontally
				[i]scaleY[/i] : The amount to scale by vertically
				[i]destMask[/i] : The pre-allocated mask to hold the scaled version
				[i]Returns[/i] : destMask if not Null, otherwise a new Mask
				
				If destMask is supplied then the scaled version is written to it.
				
				If destMask is Null then then a new mask containing the scaled version is returned.
				Please note that New is called if a destMask is not supplied, therefore a pre-allocated
				mask should always be supplied if this method is used within a main game loop.
			#End
			Method Scale:Mask(scaleX:Float, scaleY:Float, destMask:Mask = Null)
				If destMask = Null
					destMask = New Mask()
					destMask.data = New Int[ (Self.width * scaleX) * (Self.height * scaleY)]
			#If CONFIG = "debug"
				Else
					If destMask.data.Length < (Self.width * scaleX) * (Self.height * scaleY) Then Error("Supplied mask is too small.")
			#EndIf
				EndIf
				
				destMask.width = Self.width * scaleX
				destMask.height = Self.height * scaleY
				destMask.flags = Self.flags
				destMask.handleX = Self.handleX * scaleX
				destMask.handleY = Self.handleY * scaleY
				destMask.threshold = Self.threshold
				destMask.scaleX = scaleX
				destMask.scaleY = scaleY
				
				Local xStep:Float = Float(Self.width) / destMask.width
				Local yStep:Float = Float(Self.height) / destMask.height

				Local sourceX:Float = 0
				Local sourceY:Float = 0
				Local sourceOffset:Int = 0
				Local destOffset:Int = 0
				
				For Local y:Int = 0 Until destMask.height
					sourceX = 0
					sourceOffset = Int(sourceY) * Self.width
					For Local x:Int = 0 Until destMask.width
						destMask.data[destOffset + x] = Self.data[sourceOffset + Int(sourceX)]
						sourceX += xStep;
					Next
					destOffset += destMask.width
					sourceY += yStep;
				Next
				
				Return destMask
			End
			
		Public
			#Rem
				Summary: Scale the mask by the specified X and Y amounts and rotate by the specified angle
				[i]scaleX[/i] : The amount to scale by horizontally
				[i]scaleY[/i] : The amount to scale by vertically
				[i]rotation[/i] : The amount to rotate by (in degrees)
				[i]destMask[/i] : The pre-allocated mask to hold the scaled version
				[i]Returns[/i] : destMask if not Null, otherwise a new Mask
				
				If destMask is supplied then the scaled and rotated version is written to it.
				
				If destMask is Null then then a new mask containing the scaled and rotated version is returned.
				Please note that New is called if a destMask is not supplied, therefore a pre-allocated
				mask should always be supplied if this method is used within a main game loop.
			#End
			Method ScaleAndRotate:Mask(scaleX:Float, scaleY:Float, rotation:Float, destMask:Mask = Null)
				'Calculate the size required
				Local scaledWidth:Int = Self.width * scaleX
				Local scaledHeight:Int = Self.height * scaleY
				Local scaledHandleX:Int = Self.handleX * scaleX
				Local scaledHandleY:Int = Self.handleY * scaleY

				Local cosAngle:Float = Cos(rotation)
				Local sinAngle:Float = Sin(rotation)
				
				'Calculate the position of the corners of the scaled mask
				Local x1:Int = scaledHandleX + (0 - scaledHandleX) * cosAngle + (0 - scaledHandleY) * sinAngle
				Local y1:Int = scaledHandleY - (0 - scaledHandleX) * sinAngle + (0 - scaledHandleY) * cosAngle
				Local x2:Int = scaledHandleX + (0 - scaledHandleX) * cosAngle + (scaledHeight - scaledHandleY) * sinAngle
				Local y2:Int = scaledHandleY - (0 - scaledHandleX) * sinAngle + (scaledHeight - scaledHandleY) * cosAngle
				Local x3:Int = scaledHandleX + (scaledWidth - scaledHandleX) * cosAngle + (scaledHeight - scaledHandleY) * sinAngle
				Local y3:Int = scaledHandleY - (scaledWidth - scaledHandleX) * sinAngle + (scaledHeight - scaledHandleY) * cosAngle
				Local x4:Int = scaledHandleX + (scaledWidth - scaledHandleX) * cosAngle + (0 - scaledHandleY) * sinAngle
				Local y4:Int = scaledHandleY - (scaledWidth - scaledHandleX) * sinAngle + (0 - scaledHandleY) * cosAngle
				
				'Calculate the maximum and minimum values of the corners
				Local minX:Int = Min(Min(Min(x1, x2), x3), x4)
				Local maxX:Int = Max(Max(Max(x1, x2), x3), x4)
				Local minY:Int = Min(Min(Min(y1, y2), y3), y4)
				Local maxY:Int = Max(Max(Max(y1, y2), y3), y4)
				
				'If required create a new mask and allocate a large enough array to hold the data
				If destMask = Null
					destMask = New Mask()
					destMask.data = New Int[ (maxX - minX) * (maxY - minY)]
			#If CONFIG = "debug"
				Else
					If destMask.data.Length < (maxX - minX) * (maxY - minY) Then Error("Supplied mask is too small")
			#EndIf
				EndIf
				
				'Set the new size and handle
				destMask.width = maxX - minX
				destMask.height = maxY - minY
				destMask.flags = Self.flags
				destMask.handleX = scaledHandleX - minX
				destMask.handleY = scaledHandleY - minY
				destMask.threshold = Self.threshold
				destMask.scaleX = scaleX
				destMask.scaleY = scaleY
				destMask.rotation = rotation
				
				'Rotate and scale the mask
				Local xRatio:Float = 1.0 / scaleX
				Local yRatio:Float = 1.0 / scaleY
				
			    Local uCol:Float = -sinAngle * xRatio
			    Local vCol:Float = cosAngle * yRatio
			 
				Local uRow:Float = cosAngle * xRatio
			    Local vRow:Float = sinAngle * yRatio
			 
				Local rowU:Float = Self.handleX - (destMask.handleX * cosAngle + destMask.handleY * - sinAngle) * xRatio
			    Local rowV:Float = Self.handleY - (destMask.handleX * sinAngle + destMask.handleY * cosAngle) * yRatio
				Local destRowOffset:Int = 0
				
				For Local y:Int = 0 Until destMask.height
			        Local u:Float = rowU
			        Local v:Float = rowV
			 
					For Local x:Int = 0 Until destMask.width
			            If u >= 0 And v >= 0 And u < Self.width And v < Self.height
							destMask.data[destRowOffset + x] = Self.data[ (Int(v) * Self.width) + Int(u)]
			            Else
							destMask.data[destRowOffset + x] = 0
			            EndIf

			            u += uRow;
			            v += vRow;
			        Next
					
					destRowOffset += destMask.width
			 
			        rowU += uCol;
			        rowV += vCol;
			    Next
				
				Return destMask
			End
			
		Public
			#Rem
				Summary: Create a new mask with enough space to store a rotated and scaled mask with the supplied scaling factors
				If the populate flag is set to True then the MasterMask data is copied to the new Mask
				
				Please note that New is called in this method, therefore it should not be used within a main game loop.
			#End
			Method PreAllocateMask:Mask(maxScaleX:Float = 1.0, maxScaleY:Float = 1.0, populate:Bool = False)
				Local mask:Mask = New Mask()
				mask.width = Self.width
				mask.height = Self.height
				mask.flags = Self.flags
				mask.handleX = Self.handleX
				mask.handleY = Self.handleY
				mask.threshold = Self.threshold
				
				Local maxWidth:Float = Self.width * maxScaleX
				Local maxHeight:Float = Self.height * maxScaleY
				
				mask.data = New Int[maxWidth * maxWidth + maxHeight * maxHeight] ' Maximum size possible (Pythagorean hypotenuse)
				
				If populate = True
					For Local offset:Int = 0 Until Self.data.Length
						mask.data[offset] = Self.data[offset]
					Next
				EndIf
				
				Return mask
			End
			
	End

Public
	'Summary: Mask Interface for asynchronous loading
	Interface IOnLoadMaskComplete
		#Rem
			Summary: The mask has completed loading
			[i]mask[/i] : A MasterMask object that was created from the CMSK file
			[i]path[/i] : The full path of the CMSK file that has completed loading
		#End
		Method OnLoadMaskComplete:Void(mask:MasterMask, path:String)
	End
